// Invoice creation functionality
document.addEventListener('DOMContentLoaded', function() {
    // Billing type toggle
    const billingType = document.getElementById('billingType');
    const taxHeader = document.getElementById('taxHeader');
    const taxRow = document.getElementById('taxRow');
    const gstinField = document.getElementById('gstinField');
    
    if (billingType) {
        billingType.addEventListener('change', function() {
            const isGST = this.value === 'gst';
            taxHeader.style.display = isGST ? 'table-cell' : 'none';
            taxRow.style.display = isGST ? 'block' : 'none';
            gstinField.style.display = isGST ? 'block' : 'none';
            
            // Show/hide tax fields in all rows
            document.querySelectorAll('.tax-field').forEach(field => {
                field.style.display = isGST ? 'table-cell' : 'none';
            });
            
            calculateTotals();
        });
    }
    
    // Add new item row
    const addItemBtn = document.getElementById('addItem');
    const itemsBody = document.getElementById('itemsBody');
    let itemCount = 1;
    
    if (addItemBtn) {
        addItemBtn.addEventListener('click', function() {
            const newRow = document.createElement('tr');
            newRow.innerHTML = `
                <td>
                    <input type="text" name="items[${itemCount}][description]" class="form-control" placeholder="Item description">
                    <input type="hidden" name="items[${itemCount}][product_id]" value="0">
                </td>
                <td>
                    <input type="number" name="items[${itemCount}][quantity]" class="form-control quantity" value="1" min="1" step="0.01">
                </td>
                <td>
                    <input type="number" name="items[${itemCount}][price]" class="form-control price" value="0" min="0" step="0.01">
                </td>
                <td class="tax-field" style="display: ${billingType?.value === 'gst' ? 'table-cell' : 'none'};">
                    <select name="items[${itemCount}][tax_rate]" class="form-select tax-rate">
                        <option value="0">0%</option>
                        <option value="5">5%</option>
                        <option value="12">12%</option>
                        <option value="18">18%</option>
                        <option value="28">28%</option>
                    </select>
                </td>
                <td>
                    <input type="text" class="form-control item-total" value="0.00" readonly>
                </td>
                <td>
                    <button type="button" class="btn btn-danger btn-sm remove-item">
                        <i class="fas fa-times"></i>
                    </button>
                </td>
            `;
            
            itemsBody.appendChild(newRow);
            itemCount++;
            
            // Enable remove buttons for all but first row
            updateRemoveButtons();
            
            // Add event listeners to new inputs
            addEventListenersToRow(newRow);
        });
    }
    
    // Calculate totals function
    function calculateTotals() {
        let subtotal = 0;
        let taxAmount = 0;
        const isGST = billingType?.value === 'gst';
        
        document.querySelectorAll('#itemsBody tr').forEach(row => {
            const quantity = parseFloat(row.querySelector('.quantity').value) || 0;
            const price = parseFloat(row.querySelector('.price').value) || 0;
            const taxRate = parseFloat(row.querySelector('.tax-rate')?.value) || 0;
            const itemTotal = quantity * price;
            
            subtotal += itemTotal;
            
            if (isGST) {
                taxAmount += itemTotal * (taxRate / 100);
            }
            
            // Update item total
            const itemTotalInput = row.querySelector('.item-total');
            if (itemTotalInput) {
                itemTotalInput.value = itemTotal.toFixed(2);
            }
        });
        
        const totalAmount = subtotal + taxAmount;
        
        // Update summary
        const subtotalElement = document.getElementById('subtotal');
        const taxAmountElement = document.getElementById('taxAmount');
        const totalAmountElement = document.getElementById('totalAmount');
        
        if (subtotalElement) subtotalElement.textContent = '₹' + subtotal.toFixed(2);
        if (taxAmountElement) taxAmountElement.textContent = '₹' + taxAmount.toFixed(2);
        if (totalAmountElement) totalAmountElement.textContent = '₹' + totalAmount.toFixed(2);
    }
    
    // Add event listeners to a row
    function addEventListenersToRow(row) {
        const quantityInput = row.querySelector('.quantity');
        const priceInput = row.querySelector('.price');
        const taxRateSelect = row.querySelector('.tax-rate');
        
        [quantityInput, priceInput, taxRateSelect].forEach(element => {
            if (element) {
                element.addEventListener('input', calculateTotals);
            }
        });
        
        const removeBtn = row.querySelector('.remove-item');
        if (removeBtn) {
            removeBtn.addEventListener('click', function() {
                row.remove();
                calculateTotals();
                updateRemoveButtons();
            });
        }
    }
    
    // Update remove buttons state
    function updateRemoveButtons() {
        const rows = document.querySelectorAll('#itemsBody tr');
        const removeButtons = document.querySelectorAll('.remove-item');
        
        removeButtons.forEach((btn, index) => {
            btn.disabled = rows.length <= 1;
        });
    }
    
    // Initialize event listeners for existing rows
    document.querySelectorAll('#itemsBody tr').forEach(row => {
        addEventListenersToRow(row);
    });
    
    // Initial calculation
    calculateTotals();
});

// Razorpay Payment Integration
function initiateRazorpayPayment(invoiceId, amount, currency = 'INR') {
    const options = {
        key: 'YOUR_RAZORPAY_KEY_ID', // Replace with your Razorpay key
        amount: amount * 100, // Amount in paise
        currency: currency,
        name: 'SM Tech Billing',
        description: 'Invoice Payment',
        invoice_id: invoiceId,
        handler: function(response) {
            // Handle successful payment
            updateInvoiceStatus(invoiceId, 'paid', response.razorpay_payment_id);
        },
        prefill: {
            name: 'Customer Name',
            email: 'customer@example.com',
            contact: '9999999999'
        },
        theme: {
            color: '#3498db'
        }
    };
    
    const rzp = new Razorpay(options);
    rzp.open();
}

function updateInvoiceStatus(invoiceId, status, paymentId) {
    fetch(`/update-payment.php`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            invoice_id: invoiceId,
            status: status,
            payment_id: paymentId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Payment successful!');
            window.location.reload();
        } else {
            alert('Payment update failed: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Payment update failed');
    });
}

// Print invoice
function printInvoice() {
    window.print();
}

// Download invoice as PDF
function downloadPDF(invoiceId) {
    // This would typically call a server-side PDF generation script
    window.open(`download.php?id=${invoiceId}`, '_blank');
}

// Share invoice via email
function shareInvoice(invoiceId) {
    const email = prompt('Enter email address to share invoice:');
    if (email) {
        // Send AJAX request to share invoice
        fetch(`share.php?id=${invoiceId}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ email: email })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Invoice shared successfully!');
            } else {
                alert('Failed to share invoice: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Failed to share invoice');
        });
    }
}